<?php
/**
 * @copyright  Vertex. All rights reserved.  https://www.vertexinc.com/
 * @author     Mediotype                     https://www.mediotype.com/
 */

namespace Vertex\Tax\Block\Adminhtml\Config\Form\Field;

use Magento\Backend\Block\Template\Context;
use Magento\Config\Block\System\Config\Form\Field;
use Magento\Framework\Data\Form\Element\AbstractElement;
use Magento\Store\Model\ScopeInterface;
use Vertex\Tax\Model\Config;
use Vertex\Tax\Model\Config\DisableMessage;
use Vertex\Tax\Model\ConfigurationValidator;

/**
 * Displays the result of a validation against the Vertex configuration settings
 */
class VertexStatus extends Field
{
    /** @var Config */
    private $config;

    /** @var ConfigurationValidator */
    private $configurationValidator;

    /** @var DisableMessage */
    private $disableMessage;

    /**
     * @param Context $context
     * @param Config $config
     * @param ConfigurationValidator $configurationValidator
     * @param DisableMessage $disableMessage
     * @param array $data
     */
    public function __construct(
        Context $context,
        Config $config,
        ConfigurationValidator $configurationValidator,
        DisableMessage $disableMessage,
        array $data = []
    ) {
        $this->config = $config;
        $this->configurationValidator = $configurationValidator;
        $this->disableMessage = $disableMessage;

        parent::__construct($context, $data);
    }

    /**
     * Get markup showing status and/or validity of Vertex configuration
     *
     * MEQP2 Warning: Protected method. Required to override Field's _getElementHtml
     * MEQP2 Warning: Discouraged function. Required for PHP < 5.6 compatibility
     *
     * @param AbstractElement $element
     * @return string
     * @SuppressWarnings(PHPMD.UnusedFormalParameter) Necessary for override
     */
    protected function _getElementHtml(AbstractElement $element)
    {
        $store = (int) $this->getRequest()->getParam('store', 0);
        $website = (int) $this->getRequest()->getParam('website', 0);
        $scopeId = $website > 0 ? $website : $store;
        $useWebsite = $website > 0;
        $scopeType = $useWebsite ? ScopeInterface::SCOPE_WEBSITE : ScopeInterface::SCOPE_STORE;

        if (!$this->config->isVertexActive($scopeId, $scopeType)
            || !$this->config->isTaxCalculationEnabled($scopeId, $scopeType)
        ) {
            $state = 'critical';
            $status = 'Disabled';
        } else {
            $result = $this->configurationValidator->execute($scopeType, $scopeId);
            if ($result->isValid()) {
                $state = 'notice';
                $status = 'Valid';
            } else {
                $message = $result->getMessage();
                $arguments = $result->getArguments();

                $state = 'minor';
                $status = __($message, $arguments);
            }
        }
        $html = '<span class="grid-severity-' . $state . '"><span>' . $status . '</span></span>';
        if ($disableMessage = $this->disableMessage->getMessage($scopeId)) {
            $html .= '<div class="vertex__automatically-disabled-message">' . $disableMessage . '</div>';
        }

        return $html;
    }

    /**
     * Determine if the "Use default value" inheritance checkbox should be shown.
     *
     * Implementation: No.
     *
     * @param AbstractElement $element
     * @return bool
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function _isInheritCheckboxRequired(AbstractElement $element)
    {
        return false;
    }
}
